import logging
import os  # Import the 'os' module
from flask import Flask, render_template, request
import requests
import argparse
import json
from langchain.llms import OpenAI

from google.cloud import vision

# Get the current working directory
current_directory = os.getcwd()
# Create the log file path relative to the current working directory
log_file_path = os.path.join(current_directory, 'app_debug.log')

# Configure logging
logging.basicConfig(
    level=logging.DEBUG,  # Set the logging level
    format='%(asctime)s [%(levelname)s] - %(message)s',
    filename=log_file_path  # Set the log file path to the current directory
)

# instantiate the OpenAI intance
llm = OpenAI(openai_api_key=os.getenv("OPENAI_API_KEY"))

app = Flask(__name__)
app.debug = True  # Enable debug mode

@app.route("/")
def home():
    return render_template("index.html")


def get_business_card_details(imageText):
    # Prepare message for ChatGPT AI
    message = "Reformat the raw data below: \n\n" + imageText + '''
\n\nInto this json format. leave empty where values not found on raw data. respond with formatted data only no additional explanation:

{{
	"address": {{address_value}},
	"company": {{company_value}},
	"email_address": {{email_address_value}},
	"job_title": {{job_title_value}},
	"name": {{name_value}},
	"phone": {{phone_value}},
	"website": {{website_value}}
}}
'''
    return str(llm.predict(message))


@app.route("/business-card-extract", methods=['POST'])
def business_card_extract():
    # Get post requests
    requestKey = request.form['apiKey']
    imageUrl = request.form['imageUrl']

    accessKey = os.getenv("API_ACCESS_KEY")
    
    # Load the OpenAI API key from the environment variable
    if accessKey is None or accessKey == "":
        logging.error(f"API_ACCESS_KEY is not set") 
        return json.dumps({
            "success": False,
            "data": "Server Error: API_ACCESS_KEY is not set",
        }), 500
    
    if imageUrl is None or imageUrl == "":
        return json.dumps({
            "success": False,
            "data": "Invalid imageUrl parameter",
        }), 401
    
    if accessKey != requestKey:
        return json.dumps({
            "success": False,
            "data": "Access Denied!",
        }), 400
    
    client = vision.ImageAnnotatorClient()
    image = vision.Image()
    image.source.image_uri = imageUrl

    response = client.document_text_detection(image=image)

    if response.error.message:
        logging.error(f"API_ACCESS_KEY is not set")
        return json.dumps({
            "success": False,
            "data": "Server Error: {}".format(response.error.message)}), 500
    
    imageText = response.full_text_annotation.text

    data = get_business_card_details(imageText)

    # Remove escape sequences for newlines and tabs, and unescape quotes
    unescaped_string = data.replace('\\n', '').replace('\\t', '\t').replace('\\"', '"')

    try:
        # Attempt to parse the unescaped string to JSON
        parsed_json = json.loads(unescaped_string)
    except json.JSONDecodeError as e:
        # Handle JSON decoding error
        logging.error(f"Server Error: {e}")
        return json.dumps({
            "success": False,
            "data": f"Server Error: {e}"}), 500

    # Return the extraction response
    return json.dumps({
            "success": True,
            "data": parsed_json})

if __name__ == "__main__":
    app.run()